<?php

namespace App\Services\Api\Contracts\V1;

use App\Models\Contract;
use App\Models\Generator;
use App\Models\LicenseKey;
use App\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Str;

class ContractResponseService {
	public static function getInfo( $contractKey ) {
		$contract = Contract::exclude( [ 'contract_key' ] )->where( 'contract_key', $contractKey )->first();

		if ( ! $contract ) {
			return [
				'response' => [
					'code'    => 1200,
					'message' => 'Contract not found',
				]
			];
		}

		$product   = Product::exclude( [ 'algorithm', 'passphrase', 'private_key', 'require_non_expired', 'status' ] )->find( $contract->product_id );
		$generator = Generator::exclude( [ 'product_id', 'status' ] )->where( 'product_id', $contract->product_id )->first();

		if ( ! $product ) {
			return [
				'response' => [
					'code'    => 1202,
					'message' => 'Product not found',
				]
			];
		}

		if ( ! $generator ) {
			return [
				'response' => [
					'code'    => 1203,
					'message' => 'Generator not found',
				]
			];
		}

		return [
			'product_id' => $contract->product_id,
			'response'   => [
				'code'      => 1204,
				'message'   => 'Contract found',
				'contract'  => $contract,
				'product'   => $product,
				'generator' => $generator
			]
		];
	}

	public static function generate( $contractKey, $quantity ) {
		$contract = Contract::select( [ 'id', 'product_id', 'license_keys_quantity' ] )->where( 'contract_key', $contractKey )->first();

		if ( ! $contract ) {
			return [
				'response' => [
					'code'    => 1200,
					'message' => 'Contract not found',
				]
			];
		}

		if ( $contract->license_keys_count >= $contract->license_keys_quantity ) {
			return [
				'response' => [
					'code'    => 1205,
					'message' => 'License keys limit reached',
				]
			];
		}

		if ( ( $contract->license_keys_count + $quantity ) > $contract->license_keys_quantity ) {
			return [
				'response' => [
					'code'                           => 1206,
					'message'                        => 'Cannot generate this quantity',
					'contract_license_keys_quantity' => $contract->license_keys_quantity,
					'current_license_keys_count'     => $contract->license_keys_count,
				]
			];
		}

		$generator = Generator::where( 'product_id', $contract->product_id )->first();

		if ( ! $generator ) {
			return [
				'response' => [
					'code'    => 1203,
					'message' => 'Generator not found',
				]
			];
		}

		$licenseKeys = [];
		$product_id  = 0;

		if ( $generator->method == 'uuid' ) {
			for ( $generated = 0; $generated < $quantity; $generated ++ ) {
				do {
					$generatedLicenseKey = $generator->prefix . Str::uuid()->toString() . $generator->suffix;
				} while ( LicenseKey::where( 'license_key', $generatedLicenseKey )->count() > 0 );

				$licenseKey                   = new LicenseKey();
				$licenseKey->product_id       = $generator->product_id;
				$licenseKey->license_key      = $generatedLicenseKey;
				$licenseKey->activation_limit = $generator->activation_limit;
				$licenseKey->validity         = $generator->validity;
				$licenseKey->status           = 'available';
				$licenseKey->contract_id      = $contract->id;

				$licenseKey->save();
				$licenseKeys[] = $generatedLicenseKey;
				$product_id    = $generator->product_id;
			}
		} else if ( $generator->method == 'chunk-system' ) {
			for ( $generated = 0; $generated < $quantity; $generated ++ ) {
				do {
					$generatedLicenseKey = $generator->prefix . generateLicenseKey( $generator->charset, $generator->number_of_chunks, $generator->chunk_length ) . $generator->suffix;
				} while ( LicenseKey::where( 'license_key', $generatedLicenseKey )->count() > 0 );

				$licenseKey                   = new LicenseKey();
				$licenseKey->product_id       = $generator->product_id;
				$licenseKey->license_key      = $generatedLicenseKey;
				$licenseKey->activation_limit = $generator->activation_limit;
				$licenseKey->validity         = $generator->validity;
				$licenseKey->status           = 'available';
				$licenseKey->contract_id      = $contract->id;

				$licenseKey->save();
				$licenseKeys[] = $generatedLicenseKey;
				$product_id    = $generator->product_id;
			}
		} else if ( $generator->method == 'custom' ) {
			for ( $generated = 0; $generated < $quantity; $generated ++ ) {
				do {
					$generatedLicenseKey = $generator->prefix . call_user_func( $generator->function_name ) . $generator->suffix;
				} while ( LicenseKey::where( 'license_key', $generatedLicenseKey )->count() > 0 );

				$licenseKey                   = new LicenseKey();
				$licenseKey->product_id       = $generator->product_id;
				$licenseKey->license_key      = $generatedLicenseKey;
				$licenseKey->activation_limit = $generator->activation_limit;
				$licenseKey->validity         = $generator->validity;
				$licenseKey->status           = 'available';
				$licenseKey->contract_id      = $contract->id;

				$licenseKey->save();
				$licenseKeys[] = $generatedLicenseKey;
				$product_id    = $generator->product_id;
			}
		}

		return [
			'product_id' => $product_id,
			'response'   => [
				'code'        => 1300,
				'message'     => 'Contract license keys generated',
				'licenseKeys' => $licenseKeys
			]
		];
	}

	public static function destroy( $contractKey, $licenseKey ) {
		$contract = Contract::select( [ 'id' ] )->where( 'contract_key', $contractKey )->first();

		if ( ! $contract ) {
			return [
				'response' => [
					'code'    => 1200,
					'message' => 'Contract not found',
				]
			];
		}

		$licenseKey = LicenseKey::where( [ 'license_key' => $licenseKey, 'contract_id' => $contract->id ] )->delete();

		if ( $licenseKey ) {
			return [
				'response' => [
					'code'    => 1400,
					'message' => 'License key deleted',
				]
			];
		}

		return [
			'response' => [
				'code'    => 1401,
				'message' => 'License key not found',
			]
		];
	}

	public static function destroyAll( $contractKey ) {
		$contract = Contract::select( [ 'id' ] )->where( 'contract_key', $contractKey )->first();

		if ( ! $contract ) {
			return [
				'response' => [
					'code'    => 1200,
					'message' => 'Contract not found',
				]
			];
		}

		$licenseKey = LicenseKey::where( [ 'contract_id' => $contract->id ] )->delete();

		if ( $licenseKey ) {
			return [
				'response' => [
					'code'            => 1500,
					'message'         => 'License keys deleted',
					'total_destroyed' => $licenseKey
				]
			];
		}

		return [
			'response' => [
				'code'    => 1501,
				'message' => 'No license keys not found',
			]
		];
	}

	public static function respond( array $data, $code = 200 ): JsonResponse {
		$authenticationMethod = getOption( 'api', 'authenticationMethod', 'sign' );

		if ( $authenticationMethod == 'sign' ) {
			return ContractResponseService::sign( $data, $code );
		}

		return ContractResponseService::encrypt( $data, $code );
	}

	public static function sign( array $data, $code = 200 ): JsonResponse {
		$private_key    = '';
		$passphrase     = '';
		$algorithm      = '';
		$privateKeyUsed = '';

		if ( isset( $data['product_id'] ) && (int) $data['product_id'] > 0 ) {
			$product = Product::select( [ 'passphrase', 'private_key', 'algorithm' ] )->where( 'id', $data['product_id'] )->first();

			$private_key    = $product->private_key;
			$passphrase     = $product->passphrase;
			$algorithm      = $product->algorithm;
			$privateKeyUsed = 'product';
		}

		if ( $passphrase == '' || ! in_array( $algorithm, array( 1, 2, 3, 6, 7, 8, 9, 10 ) ) ) {
			$private_key    = getOption( 'api', 'privateKey', '' );
			$passphrase     = getOption( 'api', 'passphrase', '' );
			$algorithm      = getOption( 'api', 'algorithm', '' );
			$privateKeyUsed = 'global';
		}

		list( $data, $response, $signature ) = sign( $data['response'], $private_key, $passphrase, $algorithm );

		return response()->json( [
			'response'         => $data,
			'response_base64'  => $response,
			'private_key_used' => $privateKeyUsed,
			'signature'        => $signature
		], $code );
	}

	public static function encrypt( array $data, $code = 200 ): JsonResponse {
		$private_key    = '';
		$passphrase     = '';
		$algorithm      = '';
		$privateKeyUsed = '';

		if ( isset( $data['product_id'] ) && (int) $data['product_id'] > 0 ) {
			$product = Product::select( [ 'passphrase', 'private_key', 'algorithm' ] )->where( 'id', $data['product_id'] )->first();

			$private_key    = $product->private_key;
			$passphrase     = $product->passphrase;
			$algorithm      = $product->algorithm;
			$privateKeyUsed = 'product';
		}

		if ( $passphrase == '' || ! in_array( $algorithm, array( 1, 2, 3, 6, 7, 8, 9, 10 ) ) ) {
			$private_key    = getOption( 'api', 'privateKey', '' );
			$passphrase     = getOption( 'api', 'passphrase', '' );
			$algorithm      = getOption( 'api', 'algorithm', '' );
			$privateKeyUsed = 'global';
		}

		$encryptedData = encryptData( $data['response'], $private_key, $passphrase, $algorithm );

		return response()->json( [
			'response'         => $encryptedData,
			'private_key_used' => $privateKeyUsed,
		], $code );
	}
}