<?php

namespace App\Http\Livewire;

use App\Models\Contract;
use App\Services\Log;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Schema;
use Livewire\Component;
use Livewire\WithPagination;

class Contracts extends Component {
	use WithPagination;

	protected string $paginationTheme = 'bootstrap';

	public $filterSearch, $bulkAction, $bulkItems, $bulkActionStatus, $bulkActionConfirm, $updateId, $deleteId, $contractKey, $productId, $name, $information, $licenseKeysQuantity, $status, $canGetInfo, $canGenerate, $canDestroy, $canDestroyAll;

	protected $listeners = [ 'bulkAction', 'bulkActionConfirmed', 'createItem', 'updateItem' ];

	protected function getCreateRules(): array {
		return [
			'contractKey'         => 'required|string',
			'name'                => 'required|string',
			'information'         => 'required|string',
			'productId'           => 'required|exists:products,id',
			'status'              => 'required|in:active,inactive',
			'licenseKeysQuantity' => 'required|numeric|min:1',
		];
	}

	protected function getUpdateRules(): array {
		return [
			'contractKey'         => 'required|string',
			'name'                => 'required|string',
			'information'         => 'required|string',
			'productId'           => 'required|exists:products,id',
			'status'              => 'required|in:active,inactive',
			'licenseKeysQuantity' => 'required|numeric|min:1',
		];
	}

	protected function getDeleteRules(): array {
		return [
			'deleteId' => 'required',
		];
	}

	public function deleteItem(): RedirectResponse {
		$this->validate( $this->getDeleteRules() );
		$this->dispatchBrowserEvent( 'close-delete-modal' );

		Contract::find( $this->deleteId )->delete();

		return redirect()->back();
	}

	public function createItem(): RedirectResponse {
		$this->validate( $this->getCreateRules() );

		$contract = new Contract();
		$this->assignItemFields( $contract );

		Log::add( '209', Auth::id(), request()->ip(), [ 'contract-id' => $contract->id ] );

		$this->dispatchBrowserEvent( 'close-create-modal' );
		$this->resetInputs();

		return redirect()->back();
	}

	public function createItemUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->getCreateRules() );
	}

	public function updateItem(): RedirectResponse {
		$this->validate( $this->getCreateRules() );

		$contract = Contract::find( $this->updateId );

		$this->assignItemFields( $contract );

		Log::add( '210', Auth::id(), request()->ip(), [ 'contract-id' => $contract->id ] );

		$this->dispatchBrowserEvent( 'close-update-modal' );
		$this->resetInputs();

		return redirect()->back();
	}

	public function updateItemUpdated( $propertyName ): void {
		$this->validateOnly( $propertyName, $this->getUpdateRules() );
	}

	public function openCreateModal(): void {
		$this->resetInputs();
		$this->dispatchBrowserEvent( 'open-create-modal' );
	}

	public function openUpdateModal( $id ): void {
		$this->resetInputs();

		$contract = Contract::find( $id );

		$this->updateId            = $contract->id;
		$this->contractKey         = $contract->contract_key;
		$this->name                = $contract->name;
		$this->information         = $contract->information;
		$this->productId           = $contract->product_id;
		$this->licenseKeysQuantity = $contract->license_keys_quantity;
		$this->status              = $contract->status;
		$this->canGetInfo          = $contract->can_get_info;
		$this->canGenerate         = $contract->can_generate;
		$this->canDestroy          = $contract->can_destroy;
		$this->canDestroyAll       = $contract->can_destroy_all;

		$this->dispatchBrowserEvent( 'open-update-modal', [
			'productId'   => $contract->product_id,
			'productName' => $contract->product ? $contract->product->name : sprintf( __( 'Product #%d Not Found' ), $this->productId )
		] );
	}

	public function assignItemFields( Contract $contract ) {
		$contract->contract_key          = $this->contractKey;
		$contract->name                  = $this->name;
		$contract->information           = $this->information;
		$contract->product_id            = $this->productId;
		$contract->license_keys_quantity = $this->licenseKeysQuantity;
		$contract->status                = $this->status;
		$contract->can_get_info          = (bool) $this->canGetInfo;
		$contract->can_generate          = (bool) $this->canGenerate;
		$contract->can_destroy           = (bool) $this->canDestroy;
		$contract->can_destroy_all       = (bool) $this->canDestroyAll;

		$contract->save();
	}

	public function resetInputs(): void {
		$this->contractKey         = Str::uuid()->toString();
		$this->name                = '';
		$this->information         = '';
		$this->productId           = 0;
		$this->licenseKeysQuantity = 1;
		$this->status              = 'active';
		$this->canGetInfo          = true;
		$this->canGenerate         = true;
		$this->canDestroy          = true;
		$this->canDestroyAll       = false;

		$this->resetValidation();
	}

	public function bulkAction(): void {
		$this->dispatchBrowserEvent( 'open-confirm-bulk-action' );
	}

	public function bulkActionConfirmed(): void {
		if ( $this->bulkAction == 'delete' ) {
			Contract::whereIn( 'id', $this->bulkItems )->delete();

			Log::add( '215', Auth::id(), request()->ip(), [ 'contract-key-ids' => $this->bulkItems ] );

			$this->bulkAction = '';
			$this->bulkItems  = [];
		}

		if ( $this->bulkAction == 'change-status' && in_array( $this->bulkActionStatus, [ 'active', 'inactive' ] ) ) {
			Contract::whereIn( 'id', $this->bulkItems )->update( [ 'status' => $this->bulkActionStatus ] );

			Log::add( '214', Auth::id(), request()->ip(), [ 'contract-key-ids' => $this->bulkItems ] );

			$this->bulkAction       = '';
			$this->bulkActionStatus = '';
			$this->bulkItems        = [];
		}

		$this->dispatchBrowserEvent( 'close-confirm-bulk-action' );
		$this->dispatchBrowserEvent( 'reset-bulk-input' );
	}

	public function render() {
		return view( 'livewire.contracts.contracts', [
			'contracts' => Contract::where( function ( $q ) {
				if ( $this->filterSearch != '' ) {
					foreach ( Schema::getColumnListing( 'contracts' ) as $column ) {
						$q->orWhere( $column, 'like', '%' . $this->filterSearch . '%' );
					}
				}
			} )->orderBy( 'id', 'DESC' )->paginate( getOption( 'ui', 'itemsPerTable', 10 ) )
		] );
	}
}
