<?php

namespace App\Http\Livewire;

use App\Actions\RemoteDeactivate;
use App\Models\Activation;
use App\Models\Product;
use App\Services\Log;
use Exception;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Schema;

class Activations extends Component {
	protected string $paginationTheme = 'bootstrap';

	public $filterSearch, $deleteId, $itemId, $toggleActivationAction, $bulkAction, $bulkItems, $bulkActionStatus, $bulkActionConfirm;

	protected $listeners = [ 'bulkAction', 'bulkActionConfirmed', 'filterItems', 'toggleActivation' ];


	public function deleteItem(): void {
		Activation::where( 'id', $this->deleteId )->delete();

		Log::add( '160', Auth::id(), request()->ip() );

		$this->dispatchBrowserEvent( 'close-delete-modal' );
	}

	public function toggleActivation(): void {
		$this->dispatchBrowserEvent( 'close-deactivate-modal' );

		Activation::where( 'id', $this->itemId )->update( [ 'status' => $this->toggleActivationAction == 'active' ? 'inactive' : 'active' ] );

		if ( $this->toggleActivationAction == 'active' ) {
			$activation = Activation::find( $this->itemId );
			$product    = Product::find( $activation->licenseKey->product_id );

			if ( \Config::get( 'app.is_demo', false ) ) {
				$this->dispatchBrowserEvent( 'show-toast', [
					'toastType'    => 'danger',
					'toastMessage' => __( 'Remote deactivation is disabled in the demo!' )
				] );
			} else {
				try {
					RemoteDeactivate::handle( $activation, $product );

					$this->dispatchBrowserEvent( 'show-toast', [
						'toastType'    => 'success',
						'toastMessage' => __( 'Remote deactivation request sent.' )
					] );
				} catch ( Exception $e ) {
					$this->dispatchBrowserEvent( 'show-toast', [
						'toastType'    => 'danger',
						'toastMessage' => __( 'Remote deactivation request could not be sent.' )
					] );
				}
			}
		}

		Log::add( '106', Auth::id(), request()->ip(), [ 'activation-id' => $this->itemId ] );
	}

	public function resetInputs(): void {
		$this->deleteId = 0;

		$this->resetValidation();
	}

	public function bulkAction(): void {
		$this->dispatchBrowserEvent( 'open-confirm-bulk-action' );
	}

	public function bulkActionConfirmed(): void {
		if ( $this->bulkAction == 'delete' ) {
			Activation::whereIn( 'id', $this->bulkItems )->delete();

			Log::add( '104', Auth::id(), request()->ip(), [ 'activation-ids' => $this->bulkItems ] );

			$this->bulkAction = '';
			$this->bulkItems  = [];
		}

		if ( $this->bulkAction == 'change-status' && in_array( $this->bulkActionStatus, [ 'active', 'inactive' ] ) ) {
			Activation::whereIn( 'id', $this->bulkItems )->update( [ 'status' => $this->bulkActionStatus ] );

			Log::add( '105', Auth::id(), request()->ip(), [ 'activation-ids' => $this->bulkItems ] );

			$this->bulkAction       = '';
			$this->bulkActionStatus = '';
			$this->bulkItems        = [];
		}

		$this->dispatchBrowserEvent( 'close-confirm-bulk-action' );
		$this->dispatchBrowserEvent( 'reset-bulk-input' );
	}

	public function filterItems(): void {
		$this->dispatchBrowserEvent( 'filter-done' );
	}

	public function render() {
		return view( 'livewire.activations.manage', [
			'items' => Activation::where( function ( $q ) {
				if ( $this->filterSearch != '' ) {
					foreach ( Schema::getColumnListing( 'activations' ) as $column ) {
						$q->orWhere( $column, 'like', '%' . $this->filterSearch . '%' );
					}
				}
			} )->orderBy( 'id', 'DESC' )->paginate( getOption( 'ui', 'itemsPerTable', 10 ) )
		] );
	}
}
